import bpy
from ...addon.naming import FluidLabNaming
from bpy.types import PropertyGroup, UIList, Object, Context
from .common_list_methods import FLUIDLAB_CommonList
from ...libs.functions.basics import set_active_object
from ...libs.functions.modifiers import remove_modifier_from_object
from ...libs.functions.get_common_vars import get_common_vars
from bpy.props import StringProperty, IntProperty, CollectionProperty, IntProperty, BoolProperty, PointerProperty


""" Fluids SINGLE Colliders List """


class FLUIDLAB_UL_draw_fluids_single_colliders(UIList):

    def draw_item(self, context, layout, data, item, icon, active_data, active_propname, index):

        layout.use_property_split = False
        ui = get_common_vars(context, get_ui=True)

        if not item or not item.id_name:
            layout.row().prop(item, "remove", text="Clear", icon='X')
            return
        
        if item.collider is None: 
            layout.row().prop(item, "remove", text="Clear", icon='X')
            return
        
        if item.collider.name not in context.view_layer.objects: 
            layout.row().prop(item, "remove", text="Clear", icon='X')
            return

        left_sect = layout.row(align=True)

        if ui.main_modules == 'DYNAMIC_PAINT':
            left_sect.separator()
            left_sect.prop(item, "dp_canvaseable", text="")

        icon = FluidLabNaming.COLLIDER_TYPE_OB
        left_sect.prop(item, "label_txt", text="", emboss=False, icon=icon)

        right_sect = left_sect.row(align=True)
        
        # icono ojito:
        right_sect.prop(item, "hide_set", text="", toggle=True, icon='HIDE_ON' if not item.hide_set else 'HIDE_OFF')

        if ui.main_modules == 'DYNAMIC_PAINT':
            if not item.dp_without_canvas:
                clear_canvas_bt = right_sect.row(align=True) 
                clear_canvas_bt.alert = True
                clear_canvas_bt.prop(item, "dp_without_canvas", emboss=False, icon='TRASH', icon_only=True)


class FluidSingleCollidersListItem(PropertyGroup):
    
    label_txt: StringProperty(name="Name")
    id_name: StringProperty(name="ID")
    
    # para almacenar los colliders:
    collider: PointerProperty(type=Object)

    dp_canvaseable: BoolProperty(name="Canvas", description="Mark the elements that will be assigned a canvas", default=True)

    def dp_without_canvas_update(self, context):
        if self.dp_without_canvas:
            # eliminamos el canvas del item actual:
            collider_ob = self.collider
            remove_modifier_from_object(collider_ob, 'DYNAMIC_PAINT', False)

    dp_without_canvas: BoolProperty(name="Remove", description="Remove Canvas", default=True, update=dp_without_canvas_update)

    def do_remove(self, context):
        fluid_single_colliders = get_common_vars(context, get_fluid_single_colliders=True)
        fluid_single_colliders.remove_item(self.id_name)
        fluid_single_colliders.list_index = fluid_single_colliders.length-1 if not fluid_single_colliders.is_void else 0

    remove: BoolProperty(
        default=False, 
        update=do_remove
    )

    def hide_set_update(self, context):
        # Ocultamos/Desocultamos el ojito del collider:
        ob = self.collider
        ob.hide_set(not self.hide_set)

    hide_set: BoolProperty(
        name="Hide Colliders",
        description="Hide/Unhide Colliders",
        default=True, 
        update=hide_set_update
    )


class FluidSingleCollidersList(PropertyGroup, FLUIDLAB_CommonList):

    # Attributes (with or without methods):
    def list_index_update(self, context:Context) -> None:
        item = self.active
        if not item:
            return
        
        colliders_ob = item.collider 
        if not colliders_ob:
            return
        
        bpy.ops.object.select_all(action='DESELECT')

        # Se seleccionan y se hace activo el primer collider:
        colliders_ob.select_set(True)
        set_active_object(context, colliders_ob)
    
    list_index: IntProperty(name="Layer List", description="The Layer List", default=-1, update=list_index_update)
    list: CollectionProperty(type=FluidSingleCollidersListItem)

    # Fluid Colliders List Methods:
    def add_item(self, item_id:str, label_txt:str, collider_ob:Object) -> FluidSingleCollidersListItem:

        exsist = next((item.collider for item in self.list if item.collider == collider_ob), None)
        if not exsist:
            
            item = self.list.add()
            item.id_name = item_id
            item.label_txt = label_txt
            item.collider = collider_ob
                
            # seteamos el ultimo elemento como activo:
            self.list_index = self.length-1

            return item
    
    @property
    def get_all_colliders(self):
        return [item.collider for item in self.list]
        
    @property
    def get_all_dp_canvaseable(self):
        return [item for item in self.list if item.dp_canvaseable]
    
